// The Manager Kobold who supervises storage of the comm vaults for players
//
// no retrieval of messages by this object. Player objects retrieve their LWO
// (Kobold-Vault) from here. The Vaults only release messages to their
// corresponding player object

#pragma strict_types,rtt_checks

#include "/sys/player/comm.h"

/** vaults ist ein mapping, welches alle vaults enthaelt.
 *
 * Struktur: UID als Schluessel (string) und Vault (lwobject) als
 * Wert.
*/
private nosave mapping vaults = ([]);

#define MEMORY "/secure/memory"
#ifndef DEBUG
#define DEBUG(x)  if (find_player("zesstra"))\
            tell_object(find_player("zesstra"),\
                                      "kobold: "+x+"\n")
#endif

protected void create()
{
  seteuid(getuid());
  mixed tmp = ({mapping})MEMORY->Load("Kobold");
  if (mappingp(tmp)) {
    vaults = tmp;
    DEBUG("Daten aus MEMORY geladen.\n");
  }
  else {
    DEBUG("Keine Daten in MEMORY vorhanden - reinitialisiere.\n");
    if (({int})MEMORY->Save("Kobold", vaults) != 1)
      raise_error("Konnte Daten nicht im Memory ablegen.\n");
  }
}

// Einlagern von Vaults durch Spieler
public int DepositVault(lwobject "/std/player/comm_vault" vault)
{
    if (query_once_interactive(previous_object())) {
        vaults[getuid(previous_object())] = vault;
        return 1;
    }
    return 0;
}

// Abrufen eines Pointers auf den Vault eines Spielers
// Natuerlich *nur* durch den Spieler mit der richtigen UUID selber.
public lwobject "/std/player/comm_vault" RetrieveVault()
{
    if (query_once_interactive(previous_object()))
    {
        lwobject "/std/player/comm_vault" vault = vaults[getuid(previous_object())];
        // Und noch pruefen, ob auch die UUID uebereinstimmt - es soll kein
        // neu angelegter Spieler das Vault von einem aelteren bekommen.
        if (vault &&
            ({string})vault.uuid() == getuuid(previous_object()))
            return vault;
    }
    return 0;
}

// Pointer auf nen Vault vergessen.
// Im Erfolgsfall wird das Vault zurueckgeben, was wir nicht mehr speichern
// Aufruf *nur* durch den Spieler mit der richtigen UUID selber.
public lwobject "/std/player/comm_vault" ForgetVault()
{
    if (query_once_interactive(previous_object()))
    {
        lwobject "/std/player/comm_vault" vault = vaults[getuid(previous_object())];
        // Und noch pruefen, ob auch die UUID uebereinstimmt - es soll kein
        // neu angelegter Spieler das Vault von einem aelteren bekommen.
        if (vault &&
            ({string})vault.uuid() == getuuid(previous_object()))
        {
            m_delete(vaults, getuid(previous_object()));
            return vault;
        }
    }
    return 0;
}

// Prueft, ob es ein Vault fuer Spieler gibt, deren UID mit <pluid> beginnen.
// gibt im Erfolgsfall die vollstaendige UID zurueck.
// 0, wenn keine gefunden
// -1 wenn es mehrere in Frage kommende UIDs gibt.
// (match_living()-inspiriert)
public string|int find_player(string pluid)
{
    string uid;
    if (pluid in vaults)
        return pluid;  // das war einfach...
    else
    {
        // Wir muessen leider alle Vaults absuchen
        foreach(string u : vaults) {
            if (strstr(u, pluid) == 0)
            {
                // wenn pluid nicht eindeutig ist und mehrere UIDs in Frage
                // kommen, wird -1 zurueckgeben.
                if (!uid)
                    uid = u;
                else
                    return -1;
            }
        }
    }
    return uid;
}

public varargs int DepositMsg(string pluid, string msg, int msg_type,
                              string msg_action, string msg_prefix,
                              object origin = previous_object())
{
    if (!query_once_interactive(previous_object()))
        return MSG_FAILED;
    lwobject "/std/player/comm_vault" vault = vaults[pluid];
    if (!vault)
        return MSG_FAILED;

    return ({int})vault.store_msg(msg, msg_type, msg_action, msg_prefix, origin);
}

